require('dotenv').config();
const express = require('express');
const session = require('express-session');
const bodyParser = require('body-parser');
const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcryptjs');
const path = require('path');

const app = express();
const PORT = 3015; // change if needed

// ------- CONFIG -------

const OPENAI_API_KEY = process.env.OPENAI_API_KEY; // set this in your env

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

app.use(bodyParser.urlencoded({ extended: true }));
app.use(express.static('public'));
app.use(
  session({
    secret: 'super-secret-for-son-science-lab',
    resave: false,
    saveUninitialized: false
  })
);

// ------- DB SETUP -------

const db = new sqlite3.Database(path.join(__dirname, 'db.sqlite'), (err) => {
  if (err) {
    console.error('Error opening database:', err.message);
    process.exit(1);
  }
});

// Set busy timeout before any operations
db.configure('busyTimeout', 10000);

// Initialize database tables with error handling
// Add delay to ensure database is ready
setTimeout(() => {
  db.serialize(() => {
    // Try to set WAL mode (non-critical if it fails)
    db.exec(
      "PRAGMA journal_mode = WAL;",
      err => {
        if (err) {
          console.warn("PRAGMA WAL warning (non-critical):", err.message);
        }
      }
    );
    // Users table
    db.run(
      `CREATE TABLE IF NOT EXISTS users (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        username TEXT UNIQUE,
        password_hash TEXT NOT NULL,
        role TEXT NOT NULL CHECK(role IN ('father','child'))
      )`,
      (err) => {
        if (err && err.code !== 'SQLITE_BUSY') {
          console.error('Error creating users table:', err);
        }
      }
    );

  // Tasks table
  db.run(
    `CREATE TABLE IF NOT EXISTS tasks (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      title TEXT NOT NULL,
      description TEXT,
      subject TEXT,
      duration_min INTEGER DEFAULT 10,
      active INTEGER DEFAULT 1
    )`
  );

  // Progress table (with timing + status)
  db.run(
    `CREATE TABLE IF NOT EXISTS progress (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      task_id INTEGER NOT NULL,
      date TEXT NOT NULL,   -- YYYY-MM-DD
      completed INTEGER DEFAULT 0,
      started_at TEXT,
      completed_at TEXT,
      time_spent_sec INTEGER,
      status TEXT DEFAULT 'pending', -- 'pending','completed','failed'
      UNIQUE(user_id, task_id, date),
      FOREIGN KEY(user_id) REFERENCES users(id),
      FOREIGN KEY(task_id) REFERENCES tasks(id)
    )`
  );

  // Journal table
  db.run(
    `CREATE TABLE IF NOT EXISTS journals (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      date TEXT NOT NULL,   -- YYYY-MM-DD
      content TEXT NOT NULL,
      UNIQUE(user_id, date),
      FOREIGN KEY(user_id) REFERENCES users(id)
    )`
  );

  // Weekly instructions (father → child)
  db.run(
    `CREATE TABLE IF NOT EXISTS weekly_instructions (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      created_at TEXT NOT NULL,
      maths TEXT,
      science TEXT,
      coding TEXT,
      project TEXT
    )`
  );

  // Daily time stats
  db.run(
    `CREATE TABLE IF NOT EXISTS daily_stats (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      date TEXT NOT NULL,  -- YYYY-MM-DD
      minutes_maths INTEGER DEFAULT 0,
      minutes_science INTEGER DEFAULT 0,
      minutes_reading INTEGER DEFAULT 0,
      minutes_coding INTEGER DEFAULT 0,
      UNIQUE(user_id, date),
      FOREIGN KEY(user_id) REFERENCES users(id)
    )`
  );

  // Quiz questions
  db.run(
    `CREATE TABLE IF NOT EXISTS quiz_questions (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      created_at TEXT NOT NULL,
      subject TEXT,
      question TEXT NOT NULL
    )`
  );

  // Quiz answers
  db.run(
    `CREATE TABLE IF NOT EXISTS quiz_answers (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      question_id INTEGER NOT NULL,
      user_id INTEGER NOT NULL,
      date TEXT NOT NULL,
      answer TEXT NOT NULL,
      FOREIGN KEY(question_id) REFERENCES quiz_questions(id),
      FOREIGN KEY(user_id) REFERENCES users(id)
    )`
  );

  // AI feedback
  db.run(
    `CREATE TABLE IF NOT EXISTS ai_feedback (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      task_id INTEGER NOT NULL,
      date TEXT NOT NULL,
      subject TEXT,
      difficulty_score INTEGER,
      advice TEXT,
      created_at TEXT NOT NULL,
      FOREIGN KEY(user_id) REFERENCES users(id),
      FOREIGN KEY(task_id) REFERENCES tasks(id)
    )`
  );

  // AI-generated study materials (cached per day)
  db.run(
    `CREATE TABLE IF NOT EXISTS study_materials (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      user_id INTEGER NOT NULL,
      task_id INTEGER NOT NULL,
      date TEXT NOT NULL,   -- YYYY-MM-DD
      content TEXT NOT NULL, -- HTML from AI
      UNIQUE(user_id, task_id, date),
      FOREIGN KEY(user_id) REFERENCES users(id),
      FOREIGN KEY(task_id) REFERENCES tasks(id)
    )`
  );

  // Seed default data (only if no users)
  db.get('SELECT COUNT(*) AS count FROM users', (err, row) => {
    if (err) {
      console.error('Error counting users:', err);
      return;
    }
    if (row && row.count === 0) {
      console.log('Seeding default users & tasks...');
      const fatherPassHash = bcrypt.hashSync('father123', 10); // CHANGE LATER
      const childPassHash = bcrypt.hashSync('child123', 10);   // CHANGE LATER

      db.run(
        'INSERT INTO users (username, password_hash, role) VALUES (?, ?, ?)',
        ['father', fatherPassHash, 'father']
      );
      db.run(
        'INSERT INTO users (username, password_hash, role) VALUES (?, ?, ?)',
        ['child', childPassHash, 'child']
      );

      const stmt = db.prepare(
        'INSERT INTO tasks (title, description, subject, duration_min) VALUES (?, ?, ?, ?)'
      );

      stmt.run(
        '10 min – Mental Maths',
        'Practice tables and quick addition/subtraction.',
        'maths',
        10
      );
      stmt.run(
        '20 min – Khan Academy Maths',
        'Continue Arithmetic / Pre-Algebra topics.',
        'maths',
        20
      );
      stmt.run(
        '20 min – Science Video',
        'Choose one from physics/biology/chemistry/space playlists.',
        'science',
        20
      );
      stmt.run(
        '10 min – Science Reading',
        'Read book/notes related to CBSE chapter.',
        'science',
        10
      );
      stmt.run(
        '10 min – Journal Writing',
        'Write what you learned and your questions.',
        'journal',
        10
      );
      stmt.finalize();
    }
  });
  });
}, 1000);

// ------- 1-YEAR ROADMAP -------

const yearPlan = [
  {
    key: 0,
    title: "Month 1 – Foundation Basics",
    maths: "Number system, whole numbers, fractions, decimals.",
    science: "Food components, separation of substances.",
    coding: "Start Scratch coding.",
    project: "Grow a plant & measure height; simple telescope/model."
  },
  {
    key: 1,
    title: "Month 2 – Build Confidence in Maths",
    maths: "Fractions (deep), LCM & HCF, basic ratio.",
    science: "Living organisms & habitat; body & health.",
    coding: "Scratch projects; introduction to Python if comfortable.",
    project: "Lungs model; plant/animal cell chart."
  },
  {
    key: 2,
    title: "Month 3 – Pre-Algebra Preparation",
    maths: "Patterns, simple equations, pre-algebra ideas.",
    science: "Motion and measurement; light and shadows.",
    coding: "Python – variables, loops; small calculator program.",
    project: "Sundial; reflection/refraction experiments."
  },
  {
    key: 3,
    title: "Month 4 – Physics Fun",
    maths: "Geometry basics – lines, angles, triangles.",
    science: "Electricity & circuits; magnets.",
    coding: "Python – functions; small quiz program.",
    project: "Build small circuit; magnet strength experiments."
  },
  {
    key: 4,
    title: "Month 5 – Astronomy Month",
    maths: "Coordinate ideas, reading simple graphs.",
    science: "Solar system, stars, galaxies.",
    coding: "Use Python Turtle to draw constellations.",
    project: "Track moon for 30 days; solar system model."
  },
  {
    key: 5,
    title: "Month 6 – Mid-Year Revision",
    maths: "Full Class 6 revision, focus on weak areas.",
    science: "Revise all covered chapters.",
    coding: "Small flashcard or notes app.",
    project: "Windmill or water-wheel model."
  },
  {
    key: 6,
    title: "Month 7 – Logic & Thinking",
    maths: "Puzzles, brain teasers, faster mental maths.",
    science: "Water, air, environment; classification of materials.",
    coding: "Logic puzzles in code.",
    project: "Simple water filter; air-quality/observation journal."
  },
  {
    key: 7,
    title: "Month 8 – Real Problem Solving",
    maths: "Word problems (fractions, decimals, ratios), data handling.",
    science: "Weather & climate; soil & agriculture.",
    coding: "Python – basics of data handling/graphs.",
    project: "Mini weather station; soil tests."
  },
  {
    key: 8,
    title: "Month 9 – Class 7 Foundation",
    maths: "Integers, algebra basics, simple equations.",
    science: "Nutrition; heat & temperature.",
    coding: "More Python practice, small console games.",
    project: "Temperature comparison experiments; heat conduction."
  },
  {
    key: 9,
    title: "Month 10 – Chemistry & Change",
    maths: "Basic probability, more on graphs.",
    science: "Acids/bases/salts; physical vs chemical change.",
    coding: "Science quiz app in Python.",
    project: "pH indicator experiments; rusting experiments."
  },
  {
    key: 10,
    title: "Month 11 – Technology Month",
    maths: "Logical reasoning, coding-friendly maths.",
    science: "Computers, internet, AI basics; electricity usage.",
    coding: "Small utilities in Python (calculator, timer, etc.).",
    project: "Electricity-usage audit at home."
  },
  {
    key: 11,
    title: "Month 12 – Final Upgrade & Showcase",
    maths: "Prepare for Class 7: strengthen all weak topics.",
    science: "Revise whole year; connect topics together.",
    coding: "Finish a ‘capstone’ project of his choice.",
    project: "Science exhibition project; compile ‘My Science Yearbook’."
  }
];

function getCurrentPlan() {
  const now = new Date();
  const monthIndex = now.getMonth(); // 0–11
  return yearPlan[monthIndex % yearPlan.length];
}

// ------- HELPERS -------

function requireLogin(req, res, next) {
  if (!req.session.user) return res.redirect('/login');
  next();
}

function todayKey() {
  const d = new Date();
  return d.toISOString().slice(0, 10);
}

function nowIso() {
  return new Date().toISOString();
}

function getLatestWeeklyPlan(cb) {
  db.get(
    'SELECT * FROM weekly_instructions ORDER BY datetime(created_at) DESC LIMIT 1',
    [],
    (err, row) => {
      if (err) {
        console.error('Error loading weekly plan:', err);
        return cb(null);
      }
      cb(row || null);
    }
  );
}

function minutesFromSeconds(timeSec, fallbackMinutes) {
  if (!timeSec || timeSec <= 0) return fallbackMinutes || 0;
  const m = Math.round(timeSec / 60);
  return m <= 0 ? 1 : m;
}

function updateDailyStatsFromTask(userId, date, task, timeSec) {
  return new Promise((resolve, reject) => {
    const minutes = minutesFromSeconds(timeSec, task.duration_min || 10);

    // Decide which column to update based on subject
    let field = null;
    const subj = (task.subject || '').toLowerCase();

    console.log(`[updateDailyStatsFromTask] Subject: "${task.subject}" (lowercase: "${subj}")`);

    if (subj.includes('math')) field = 'minutes_maths';
    else if (subj.includes('science')) field = 'minutes_science';
    else if (subj.includes('code') || subj.includes('computer')) field = 'minutes_coding';
    else if (subj.includes('read') || subj.includes('journal') || subj.includes('english')) {
      field = 'minutes_reading';
    }

    if (!field) {
      // Unknown subject → do nothing
      console.log(`[updateDailyStatsFromTask] Unknown subject "${task.subject}" - skipping update`);
      return resolve();
    }

    const vals = {
      minutes_maths: 0,
      minutes_science: 0,
      minutes_reading: 0,
      minutes_coding: 0
    };
    vals[field] = minutes;

    console.log(`[updateDailyStatsFromTask] Updating ${field} with ${minutes} minutes`);

    db.run(
      `INSERT INTO daily_stats (
          user_id, date,
          minutes_maths, minutes_science, minutes_reading, minutes_coding
        )
        VALUES (?, ?, ?, ?, ?, ?)
        ON CONFLICT(user_id, date)
        DO UPDATE SET
          minutes_maths   = minutes_maths   + excluded.minutes_maths,
          minutes_science = minutes_science + excluded.minutes_science,
          minutes_reading = minutes_reading + excluded.minutes_reading,
          minutes_coding  = minutes_coding  + excluded.minutes_coding`,
      [
        userId,
        date,
        vals.minutes_maths,
        vals.minutes_science,
        vals.minutes_reading,
        vals.minutes_coding
      ],
      err => {
        if (err) {
          console.error('[updateDailyStatsFromTask] Error updating daily_stats:', err);
          return reject(err);
        } else {
          console.log(`[updateDailyStatsFromTask] Successfully updated ${field} = ${minutes} minutes`);
          
          // Verify the update by reading back from DB
          db.get(
            'SELECT * FROM daily_stats WHERE user_id = ? AND date = ?',
            [userId, date],
            (verifyErr, verifyRow) => {
              if (verifyErr) {
                console.error('[updateDailyStatsFromTask] Error verifying update:', verifyErr);
              } else {
                console.log('[updateDailyStatsFromTask] Verification - Current stats:', verifyRow);
              }
              resolve(verifyRow);
            }
          );
        }
      }
    );
  });
}

async function callOpenAI(prompt, systemMessage = null) {
  const apiKey = (OPENAI_API_KEY || '').trim();
  
  console.log('[OpenAI] Checking API key...');
  console.log('[OpenAI] API key exists:', !!apiKey);
  console.log('[OpenAI] API key length:', apiKey ? apiKey.length : 0);
  console.log('[OpenAI] API key starts with sk-:', apiKey ? apiKey.startsWith('sk-') : false);
  
  if (!apiKey || apiKey === 'your-api-key-here') {
    console.warn('[OpenAI] API key not configured');
    return "AI key not configured yet. Try to split the task into smaller steps, practice similar examples, and ask for help when you feel stuck.";
  }

  const defaultSystemMessage = 'You are a gentle tutor for a 13-year-old who is weak in maths but loves science. You analyze task time and difficulty and give short, practical suggestions.';
  const systemContent = systemMessage || defaultSystemMessage;

  try {
    console.log('[OpenAI] Making API call...');
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${apiKey}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        model: 'gpt-4o-mini',
        messages: [
          {
            role: 'system',
            content: systemContent
          },
          { role: 'user', content: prompt }
        ],
        max_tokens: 500
      })
    });

    if (!response.ok) {
      const errorData = await response.json().catch(() => ({ error: 'Unknown error' }));
      console.error('[OpenAI] API error:', response.status, JSON.stringify(errorData));
      throw new Error(`OpenAI API error: ${response.status} - ${JSON.stringify(errorData)}`);
    }

    const data = await response.json();
    if (!data.choices || !data.choices[0] || !data.choices[0].message) {
      console.error('[OpenAI] Unexpected response format:', JSON.stringify(data));
      throw new Error('Unexpected response format from OpenAI');
    }
    
    const content = data.choices[0].message.content.trim();
    console.log('[OpenAI] Successfully received response');
    return content;
  } catch (error) {
    console.error('[OpenAI] Error calling API:', error.message);
    if (error.stack) {
      console.error('[OpenAI] Stack:', error.stack);
    }
    throw error;
  }
}

function classifyDifficulty(timeSec, expectedSec, status) {
  if (status === 'failed') return 5;
  if (!timeSec || !expectedSec) return 3;
  const ratio = timeSec / expectedSec;
  if (ratio <= 0.7) return 1;
  if (ratio <= 1.2) return 2;
  if (ratio <= 1.8) return 3;
  if (ratio <= 2.5) return 4;
  return 5;
}

function getExpectedTimeSec(task) {
  const min = task.duration_min || 10;
  return min * 60;
}

// Create or load AI-generated study material for a task
async function getOrCreateStudyMaterial(userId, taskId) {
  const date = todayKey();

  return new Promise((resolve, reject) => {
    db.get(
      'SELECT content FROM study_materials WHERE user_id = ? AND task_id = ? AND date = ?',
      [userId, taskId, date],
      async (err, row) => {
        if (err) return reject(err);
        if (row && row.content) {
          return resolve(row.content); // already generated for today
        }

        // Need to generate new material
        db.get('SELECT * FROM tasks WHERE id = ?', [taskId], async (err2, task) => {
          if (err2) return reject(err2);
          if (!task) return reject(new Error('Task not found'));

          const prompt = `
You are creating study material for a 13-year-old boy in Kerala (CBSE Class 6) who is weak in maths but loves science and computers.

Task title: "${task.title}"
Subject: ${task.subject || 'general'}

Create SHORT, focused HTML content only (no <html>, <body> tags). Structure:

<h3>1. Quick Concept Summary</h3>
- 3–6 bullet points explaining the idea in very simple English.

<h3>2. Worked Examples</h3>
- 2–4 worked examples with step-by-step explanation.
- For maths: show each step clearly.
- For science: clear explanation of process.

<h3>3. Practice Questions</h3>
- 6–10 practice questions, numbered. No answers.

<h3>4. Small Challenge / Real-life Connection</h3>
- 1 short activity or question connecting to real life.
`;

          let material;
          try {
            console.log(`[Study Material] Generating for task ${taskId}, user ${userId}`);
            const systemMsg = 'You are creating educational study material for a 13-year-old boy in Kerala (CBSE Class 6) who is weak in maths but loves science and computers. Create clear, simple, and encouraging content.';
            material = await callOpenAI(prompt, systemMsg);
            console.log(`[Study Material] Successfully generated ${material.length} characters`);
          } catch (e) {
            console.error('[Study Material] Error generating study material via AI:', e.message);
            console.error('[Study Material] Full error:', e);
            material =
              '<p>Study material could not be generated right now. Please revise your textbook or notes for this topic and practice a few similar problems.</p>';
          }

          // Try to cache it. If DB is busy, just return material anyway.
          db.run(
            `INSERT INTO study_materials (user_id, task_id, date, content)
             VALUES (?, ?, ?, ?)
             ON CONFLICT(user_id, task_id, date)
             DO UPDATE SET content = excluded.content`,
            [userId, taskId, date, material],
            err3 => {
              if (err3) {
                if (err3.code === 'SQLITE_BUSY') {
                  console.warn('DB busy while saving study_materials, returning material anyway.');
                  return resolve(material);
                }
                console.error('Error saving study material:', err3);
                // Still return the material so page works
                return resolve(material);
              }
              resolve(material);
            }
          );
        });
      }
    );
  });
}

async function autoAssignRemedialTask(userId, task) {
  const title = `Practice: ${task.title}`;
  const description =
    (task.description || '') +
    ' (extra practice assigned because previous attempt was hard.)';

  await new Promise((resolve) => {
    db.run(
      `INSERT INTO tasks (title, description, subject, duration_min, active)
       VALUES (?, ?, ?, ?, 1)`,
      [title, description, task.subject, task.duration_min || 10],
      err => {
        if (err) {
          console.error('Error creating remedial task (can be ignored if duplicate):', err.message);
        }
        resolve();
      }
    );
  });
}

async function analyzeTaskWithAI(userId, taskId, date) {
  const task = await new Promise((resolve, reject) => {
    db.get('SELECT * FROM tasks WHERE id = ?', [taskId], (err, row) => {
      if (err) return reject(err);
      resolve(row);
    });
  });

  const progress = await new Promise((resolve, reject) => {
    db.get(
      'SELECT * FROM progress WHERE user_id = ? AND task_id = ? AND date = ?',
      [userId, taskId, date],
      (err, row) => {
        if (err) return reject(err);
        resolve(row);
      }
    );
  });

  if (!task || !progress) return;

  const expectedSec = getExpectedTimeSec(task);
  const diffScore = classifyDifficulty(progress.time_spent_sec, expectedSec, progress.status);

  const history = await new Promise((resolve, reject) => {
    db.all(
      `SELECT p.date, p.status, p.time_spent_sec
       FROM progress p
       JOIN tasks t ON p.task_id = t.id
       WHERE p.user_id = ? AND t.subject = ?
       ORDER BY p.date DESC
       LIMIT 5`,
      [userId, task.subject],
      (err, rows) => {
        if (err) return reject(err);
        resolve(rows || []);
      }
    );
  });

  const historyText = history
    .map(h => `${h.date}: status=${h.status}, time=${h.time_spent_sec || 'null'} sec`)
    .join('\n');

  const prompt = `
Student: 13-year-old, weak in maths, motivated to become a scientist.
Task: "${task.title}" (subject: ${task.subject})
Expected time: ~${Math.round(expectedSec / 60)} minutes
Actual status today: ${progress.status}
Actual time: ${progress.time_spent_sec ? Math.round(progress.time_spent_sec / 60) : 'unknown'} minutes

Recent history for this subject:
${historyText || 'No previous history.'}

Give:
1. Very short difficulty rating (easy / okay / hard).
2. 2–3 simple, friendly suggestions to improve next time.
3. If he is struggling, suggest 1 specific type of similar practice task he should do.
Use simple English.
`;

  const advice = await callOpenAI(prompt);

  await new Promise((resolve, reject) => {
    db.run(
      `INSERT INTO ai_feedback (user_id, task_id, date, subject, difficulty_score, advice, created_at)
       VALUES (?, ?, ?, ?, ?, ?, datetime('now'))`,
      [userId, taskId, date, task.subject, diffScore, advice],
      err => {
        if (err) return reject(err);
        resolve();
      }
    );
  });

  if (progress.status === 'failed' || diffScore >= 4) {
    await autoAssignRemedialTask(userId, task);
  }
}

// ------- ROUTES -------

app.get('/', (req, res) => {
  if (!req.session.user) return res.redirect('/login');
  if (req.session.user.role === 'child') return res.redirect('/dashboard/child');
  return res.redirect('/dashboard/father');
});

// LOGIN
app.get('/login', (req, res) => {
  try {
    if (req.session.user) return res.redirect('/');
    res.render('login', { error: null });
  } catch (e) {
    console.error('Error rendering login page:', e);
    res.status(500).send('Error loading login page. Please try again.');
  }
});

app.post('/login', (req, res) => {
  try {
    const { username, password } = req.body;

    if (!username || !password) {
      return res.render('login', { error: 'Username and password required' });
    }

    db.get('SELECT * FROM users WHERE username = ?', [username], (err, user) => {
      if (err) {
        console.error('Login DB error:', err);
        try {
          return res.render('login', { error: 'Internal error. Please try again.' });
        } catch (renderErr) {
          console.error('Error rendering login with error:', renderErr);
          return res.status(500).send('Database error. Please try again later.');
        }
      }
      if (!user) {
        try {
          return res.render('login', { error: 'Invalid username or password' });
        } catch (renderErr) {
          console.error('Error rendering login:', renderErr);
          return res.status(500).send('Error loading page.');
        }
      }
      
      try {
        const match = bcrypt.compareSync(password, user.password_hash);
        if (!match) {
          try {
            return res.render('login', { error: 'Invalid username or password' });
          } catch (renderErr) {
            console.error('Error rendering login:', renderErr);
            return res.status(500).send('Error loading page.');
          }
        }
        req.session.user = { id: user.id, username: user.username, role: user.role };
        res.redirect('/');
      } catch (bcryptErr) {
        console.error('Bcrypt error:', bcryptErr);
        try {
          return res.render('login', { error: 'Authentication error. Please try again.' });
        } catch (renderErr) {
          console.error('Error rendering login:', renderErr);
          return res.status(500).send('Authentication error.');
        }
      }
    });
  } catch (e) {
    console.error('Login route error:', e);
    res.render('login', { error: 'An error occurred. Please try again.' });
  }
});

app.get('/logout', (req, res) => {
  req.session.destroy(() => {
    res.redirect('/login');
  });
});

// CHILD DASHBOARD
app.get('/dashboard/child', requireLogin, (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const date = todayKey();

  db.all('SELECT * FROM tasks WHERE active = 1', [], (err, tasks) => {
    if (err) {
      console.error(err);
      return res.send('Error loading tasks');
    }

    db.all(
      'SELECT task_id, completed FROM progress WHERE user_id = ? AND date = ?',
      [userId, date],
      (err2, rows) => {
        if (err2) {
          console.error(err2);
          return res.send('Error loading progress');
        }

        const progressMap = {};
        rows.forEach(r => {
          progressMap[r.task_id] = r.completed === 1;
        });

        db.get(
          'SELECT content FROM journals WHERE user_id = ? AND date = ?',
          [userId, date],
          (err3, journalRow) => {
            if (err3) console.error(err3);

            db.get(
              'SELECT * FROM daily_stats WHERE user_id = ? AND date = ?',
              [userId, date],
              (err4, statsRow) => {
                if (err4) console.error(err4);
                console.log(`[DASHBOARD] Loading daily_stats for user ${userId}, date ${date}:`, statsRow);

                getLatestWeeklyPlan(weeklyPlan => {
                  db.all(
                    'SELECT * FROM quiz_questions ORDER BY datetime(created_at) DESC LIMIT 3',
                    [],
                    (err5, quizQuestions) => {
                      if (err5) {
                        console.error(err5);
                        quizQuestions = [];
                      }

                      db.all(
                        `SELECT af.date, t.subject, t.title, af.advice
                         FROM ai_feedback af
                         JOIN tasks t ON af.task_id = t.id
                         WHERE af.user_id = ?
                         ORDER BY af.id DESC
                         LIMIT 3`,
                        [userId],
                        (err6, aiRows) => {
                          if (err6) {
                            console.error(err6);
                            aiRows = [];
                          }

                          res.render('dashboard_child', {
                            user: req.session.user,
                            date,
                            tasks,
                            progressMap,
                            todayJournal: journalRow ? journalRow.content : '',
                            currentPlan: getCurrentPlan(),
                            weeklyPlan,
                            dailyStats: statsRow || null,
                            quizQuestions: quizQuestions || [],
                            aiFeedback: aiRows || []
                          });
                        }
                      );
                    }
                  );
                });
              }
            );
          }
        );
      }
    );
  });
});

// TASK START
app.post('/tasks/start', requireLogin, (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const { task_id } = req.body;
  const date = todayKey();
  console.log(`[TASK START] User ${userId}, Task ${task_id}, Date ${date}`);

  db.run(
    `INSERT INTO progress (user_id, task_id, date, completed, started_at, status)
     VALUES (?, ?, ?, 0, ?, 'pending')
     ON CONFLICT(user_id, task_id, date)
     DO UPDATE SET started_at = excluded.started_at, status = 'pending'`,
    [userId, task_id, date, nowIso()],
    err => {
      if (err) console.error(err);
      res.redirect(`/task/${task_id}`);
    }
  );
});

// TASK COMPLETE / FAILED
app.post('/tasks/complete', requireLogin, async (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const { task_id, status } = req.body;
  const date = todayKey();
  const endTime = new Date();
  console.log(`[TASK COMPLETE] User ${userId}, Task ${task_id}, Status ${status}, Date ${date}`);

  db.get(
    'SELECT started_at FROM progress WHERE user_id = ? AND task_id = ? AND date = ?',
    [userId, task_id, date],
    async (err, row) => {
      if (err) {
        console.error(err);
        return res.redirect('/dashboard/child');
      }

      let timeSec = null;
      let startedAt = row && row.started_at ? new Date(row.started_at) : null;
      if (startedAt) {
        timeSec = Math.round((endTime - startedAt) / 1000);
        console.log(`[TASK COMPLETE] Time calculated: ${timeSec} seconds (${Math.round(timeSec / 60)} minutes)`);
      } else {
        console.log(`[TASK COMPLETE] No started_at found - will use task duration as fallback`);
      }

      db.run(
        `INSERT INTO progress (user_id, task_id, date, completed, status, started_at, completed_at, time_spent_sec)
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)
         ON CONFLICT(user_id, task_id, date)
         DO UPDATE SET
           completed = excluded.completed,
           status = excluded.status,
           completed_at = excluded.completed_at,
           time_spent_sec = excluded.time_spent_sec`,
        [
          userId,
          task_id,
          date,
          status === 'completed' ? 1 : 0,
          status,
          row ? row.started_at : nowIso(),
          endTime.toISOString(),
          timeSec
        ],
        async err2 => {
          if (err2) {
            console.error('[TASK COMPLETE] Error saving progress:', err2);
            return res.redirect('/dashboard/child');
          }

          // Load task to know subject + duration, then update daily_stats
          db.get('SELECT * FROM tasks WHERE id = ?', [task_id], async (err3, task) => {
            if (err3) {
              console.error('[TASK COMPLETE] Error loading task:', err3);
            }
            if (task) {
              console.log(`[TASK COMPLETE] Task loaded: ${task.title}, Subject: ${task.subject}, Duration: ${task.duration_min} min`);
              const minutesToAdd = minutesFromSeconds(timeSec, task.duration_min || 10);
              console.log(`[TASK COMPLETE] Will add ${minutesToAdd} minutes to daily_stats`);
              
              // Wait for the database update to complete before redirecting
              try {
                await updateDailyStatsFromTask(userId, date, task, timeSec);
                console.log(`[DAILY STATS] Successfully updated ${task.subject} with ${minutesToAdd} minutes`);
              } catch (updateErr) {
                console.error('[DAILY STATS] Error updating daily stats:', updateErr);
              }
            } else {
              console.error('[TASK COMPLETE] Task not found for id:', task_id);
            }

            // Optional: AI analysis (keep your existing call if you want)
            try {
              await analyzeTaskWithAI(userId, task_id, date);
            } catch (e) {
              console.error('AI analysis failed:', e);
            }

            res.redirect('/dashboard/child');
          });
        }
      );
    }
  );
});

// VIEW STUDY MATERIAL FOR A TASK
app.get('/task/:id', requireLogin, async (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const taskId = parseInt(req.params.id, 10);
  const date = todayKey();

  try {
    // Load task
    const task = await new Promise((resolve, reject) => {
      db.get('SELECT * FROM tasks WHERE id = ?', [taskId], (err, row) => {
        if (err) return reject(err);
        resolve(row);
      });
    });

    if (!task) return res.send('Task not found');

    // Ensure progress row exists (if someone opens URL directly)
    db.run(
      `INSERT INTO progress (user_id, task_id, date, completed, started_at, status)
       VALUES (?, ?, ?, 0, ?, 'pending')
       ON CONFLICT(user_id, task_id, date)
       DO UPDATE SET status = 'pending'`,
      [userId, taskId, date, nowIso()],
      err => {
        if (err) console.error('Error ensuring progress row:', err);
      }
    );

    // Get or create AI study material
    const material = await getOrCreateStudyMaterial(userId, taskId);

    // Extract Practice Questions from AI-generated content
    let questions = [];
    try {
      // Look for "3. Practice Questions" section
      const practiceSection = material.split(/<h3>3\.\s*Practice Questions<\/h3>/i)[1] || 
                             material.split(/3\.\s*Practice Questions/i)[1] || 
                             '';
      
      if (practiceSection) {
        // Extract questions using regex - matches numbered questions ending with ?
        // Pattern: number followed by period/paren, then question text ending with ?
        const regex = /(?:\d+[\.\)]\s*)([^?]+[?])/g;
        let match;
        while ((match = regex.exec(practiceSection)) !== null) {
          const questionText = match[1].trim();
          // Clean up HTML tags from question
          const cleanQuestion = questionText.replace(/<[^>]*>/g, '').trim();
          if (cleanQuestion && cleanQuestion.length > 0) {
            questions.push({ question: cleanQuestion });
          }
        }
      }
    } catch (extractErr) {
      console.error('Error extracting questions:', extractErr);
    }

    res.render('task_view', {
      user: req.session.user,
      task,
      date,
      material,
      questions
    });
  } catch (e) {
    console.error('Error in /task route:', e);
    res.send('Error loading study material.');
  }
});

// SUBMIT PRACTICE QUESTION ANSWERS
app.post('/task/:id/answers', requireLogin, async (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const taskId = parseInt(req.params.id, 10);
  const date = todayKey();

  try {
    const total = parseInt(req.body.total, 10);
    let results = [];

    for (let i = 0; i < total; i++) {
      const question = req.body[`question_${i}`];
      const answer = req.body[`answer_${i}`];
      if (question && answer) {
        results.push({ question, answer });
      }
    }

    if (results.length === 0) {
      return res.redirect(`/task/${taskId}`);
    }

    // Get AI feedback on answers
    const prompt = `
You are a helpful CBSE maths tutor for a 13-year-old student. Check the following answers and provide gentle, encouraging feedback.

${results.map((r, idx) => `Question ${idx + 1}: ${r.question}\nStudent's Answer: ${r.answer}`).join('\n\n')}

For each answer:
- If correct: Say "Correct! Well done!"
- If incorrect: Explain the mistake gently and show the correct answer
- Be encouraging and supportive
- Keep it short and simple
`;

    let aiFeedback = '';
    try {
      const systemMsg = 'You are a gentle, encouraging maths tutor for a 13-year-old student. Provide clear, supportive feedback.';
      aiFeedback = await callOpenAI(prompt, systemMsg);
    } catch (e) {
      console.error('Error getting AI feedback on answers:', e);
      aiFeedback = 'AI could not evaluate answers right now. Please check your answers with a teacher or parent.';
    }

    // Save feedback to database
    db.run(
      `INSERT INTO ai_feedback (user_id, task_id, date, subject, advice, created_at)
       VALUES (?, ?, ?, ?, ?, datetime('now'))`,
      [userId, taskId, date, 'practice', aiFeedback],
      err => {
        if (err) console.error('Error saving answer feedback:', err);
      }
    );

    // Render results page
    res.render('task_answers', {
      user: req.session.user,
      taskId,
      results,
      aiFeedback,
      date
    });
  } catch (e) {
    console.error('Error in /task/:id/answers route:', e);
    res.send('Error processing answers. Please try again.');
  }
});

// JOURNAL SAVE
app.post('/journal/save', requireLogin, (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const { content } = req.body;
  const date = todayKey();

  db.run(
    `INSERT INTO journals (user_id, date, content)
     VALUES (?, ?, ?)
     ON CONFLICT(user_id, date)
     DO UPDATE SET content = excluded.content`,
    [userId, date, content],
    err => {
      if (err) console.error(err);
      res.redirect('/dashboard/child');
    }
  );
});

// DAILY STATS SAVE
app.post('/daily-stats/save', requireLogin, (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const date = todayKey();
  const { minutes_maths, minutes_science, minutes_reading, minutes_coding } = req.body;

  const mMaths = parseInt(minutes_maths || '0', 10);
  const mSci = parseInt(minutes_science || '0', 10);
  const mRead = parseInt(minutes_reading || '0', 10);
  const mCode = parseInt(minutes_coding || '0', 10);

  db.run(
    `INSERT INTO daily_stats (user_id, date, minutes_maths, minutes_science, minutes_reading, minutes_coding)
     VALUES (?, ?, ?, ?, ?, ?)
     ON CONFLICT(user_id, date)
     DO UPDATE SET
       minutes_maths = excluded.minutes_maths,
       minutes_science = excluded.minutes_science,
       minutes_reading = excluded.minutes_reading,
       minutes_coding = excluded.minutes_coding`,
    [userId, date, mMaths, mSci, mRead, mCode],
    err => {
      if (err) console.error('Error saving daily stats:', err);
      res.redirect('/dashboard/child');
    }
  );
});

// QUIZ ANSWER (child)
app.post('/quiz/answer', requireLogin, (req, res) => {
  if (req.session.user.role !== 'child') return res.redirect('/');
  const userId = req.session.user.id;
  const { question_id, answer } = req.body;
  const date = todayKey();

  if (!question_id || !answer) {
    return res.redirect('/dashboard/child');
  }

  db.run(
    `INSERT INTO quiz_answers (question_id, user_id, date, answer)
     VALUES (?, ?, ?, ?)`,
    [question_id, userId, date, answer],
    err => {
      if (err) console.error('Error saving quiz answer:', err);
      res.redirect('/dashboard/child');
    }
  );
});

// FATHER DASHBOARD
app.get('/dashboard/father', requireLogin, (req, res) => {
  if (req.session.user.role !== 'father') return res.redirect('/');

  db.get('SELECT id, username FROM users WHERE role = "child" LIMIT 1', (err, child) => {
    if (err || !child) {
      console.error(err || 'No child user');
      return res.send('No child user found. Check DB.');
    }

    db.all(
      `SELECT p.date,
              COUNT(*) AS total_tasks,
              SUM(p.completed) AS completed_tasks
       FROM progress p
       WHERE p.user_id = ?
       GROUP BY p.date
       ORDER BY p.date DESC
       LIMIT 30`,
      [child.id],
      (err2, progressRows) => {
        if (err2) {
          console.error(err2);
          return res.send('Error loading progress');
        }

        let streakDays = 0;
        const byDate = {};
        progressRows.forEach(r => { byDate[r.date] = r; });

        let current = new Date();
        while (true) {
          const key = current.toISOString().slice(0, 10);
          const row = byDate[key];
          if (row && row.completed_tasks > 0) {
            streakDays++;
            current.setDate(current.getDate() - 1);
          } else {
            break;
          }
        }

        db.all(
          `SELECT date, minutes_maths, minutes_science, minutes_reading, minutes_coding
           FROM daily_stats
           WHERE user_id = ?
           ORDER BY date DESC
           LIMIT 14`,
          [child.id],
          (err3, statsRows) => {
            if (err3) {
              console.error(err3);
              return res.send('Error loading time stats');
            }

            db.all(
              'SELECT date, content FROM journals WHERE user_id = ? ORDER BY date DESC LIMIT 10',
              [child.id],
              (err4, journals) => {
                if (err4) {
                  console.error(err4);
                  return res.send('Error loading journals');
                }

                getLatestWeeklyPlan(weeklyPlan => {
                  db.all(
                    `SELECT q.id, q.subject, q.question,
                            a.answer, a.date AS answer_date
                     FROM quiz_questions q
                     LEFT JOIN quiz_answers a
                       ON q.id = a.question_id AND a.user_id = ?
                     ORDER BY datetime(q.created_at) DESC
                     LIMIT 10`,
                    [child.id],
                    (err5, quizRows) => {
                      if (err5) {
                        console.error(err5);
                        quizRows = [];
                      }

                      db.all(
                        `SELECT af.date, t.subject, t.title, af.difficulty_score, af.advice
                         FROM ai_feedback af
                         JOIN tasks t ON af.task_id = t.id
                         WHERE af.user_id = ?
                         ORDER BY datetime(af.created_at) DESC
                         LIMIT 20`,
                        [child.id],
                        (err6, aiRows) => {
                          if (err6) {
                            console.error(err6);
                            aiRows = [];
                          }

                          db.all(
                            `SELECT t.subject,
                                    p.date,
                                    AVG(p.time_spent_sec) AS avg_time_sec,
                                    SUM(CASE WHEN p.status = 'completed' THEN 1 ELSE 0 END) AS completed_count,
                                    SUM(CASE WHEN p.status = 'failed' THEN 1 ELSE 0 END) AS failed_count
                             FROM progress p
                             JOIN tasks t ON p.task_id = t.id
                             WHERE p.user_id = ?
                             GROUP BY t.subject, p.date
                             ORDER BY p.date ASC`,
                            [child.id],
                            (err7, perfRows) => {
                              if (err7) {
                                console.error(err7);
                                perfRows = [];
                              }

                              res.render('dashboard_father', {
                                user: req.session.user,
                                child,
                                progressRows,
                                journals,
                                streakDays,
                                currentPlan: getCurrentPlan(),
                                yearPlan,
                                weeklyPlan,
                                statsRows,
                                quizRows,
                                aiRows,
                                perfRows
                              });
                            }
                          );
                        }
                      );
                    }
                  );
                });
              }
            );
          }
        );
      }
    );
  });
});

// FATHER: WEEKLY PLAN
app.post('/father/weekly-plan', requireLogin, (req, res) => {
  if (req.session.user.role !== 'father') return res.redirect('/');
  const { maths, science, coding, project } = req.body;

  db.run(
    `INSERT INTO weekly_instructions (created_at, maths, science, coding, project)
     VALUES (datetime('now'), ?, ?, ?, ?)`,
    [maths || '', science || '', coding || '', project || ''],
    err => {
      if (err) console.error('Error saving weekly plan:', err);
      res.redirect('/dashboard/father');
    }
  );
});

// FATHER: QUIZ QUESTIONS
app.post('/father/quiz', requireLogin, (req, res) => {
  if (req.session.user.role !== 'father') return res.redirect('/');
  const { q1_subject, q1_text, q2_subject, q2_text, q3_subject, q3_text } = req.body;

  const toInsert = [];
  if (q1_text && q1_text.trim()) {
    toInsert.push({ subject: q1_subject || '', text: q1_text.trim() });
  }
  if (q2_text && q2_text.trim()) {
    toInsert.push({ subject: q2_subject || '', text: q2_text.trim() });
  }
  if (q3_text && q3_text.trim()) {
    toInsert.push({ subject: q3_subject || '', text: q3_text.trim() });
  }

  if (toInsert.length === 0) return res.redirect('/dashboard/father');

  const stmt = db.prepare(
    'INSERT INTO quiz_questions (created_at, subject, question) VALUES (datetime("now"), ?, ?)'
  );

  toInsert.forEach(q => {
    stmt.run([q.subject, q.text]);
  });

  stmt.finalize(err => {
    if (err) console.error('Error inserting quiz questions:', err);
    res.redirect('/dashboard/father');
  });
});

// ROADMAP VIEW
app.get('/roadmap', requireLogin, (req, res) => {
  res.render('roadmap', {
    user: req.session.user,
    yearPlan
  });
});

// ------- START SERVER -------

app.listen(PORT, () => {
  console.log(`Son Science Lab running at http://localhost:${PORT}`);
});
