require('dotenv').config();
const OPENAI_API_KEY = process.env.OPENAI_API_KEY;

console.log('=== Testing OpenAI API Key ===\n');

// Check if key is loaded
if (!OPENAI_API_KEY) {
  console.log('❌ ERROR: OPENAI_API_KEY not found in environment');
  console.log('Make sure .env file exists and contains: OPENAI_API_KEY=sk-...');
  process.exit(1);
}

const apiKey = OPENAI_API_KEY.trim();
console.log('✓ API Key found in .env');
console.log(`  Length: ${apiKey.length} characters`);
console.log(`  Starts with sk-: ${apiKey.startsWith('sk-')}`);
console.log(`  First 10 chars: ${apiKey.substring(0, 10)}...`);
console.log('');

// Test API call
console.log('Testing API call...');
console.log('');

async function testOpenAI() {
  try {
    const response = await fetch('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Authorization': `Bearer ${apiKey}`,
        'Content-Type': 'application/json'
      },
      body: JSON.stringify({
        model: 'gpt-4o-mini',
        messages: [
          {
            role: 'user',
            content: 'Say "Hello, API test successful!" and nothing else.'
          }
        ],
        max_tokens: 20
      })
    });

    console.log(`Response status: ${response.status} ${response.statusText}`);

    if (!response.ok) {
      const errorData = await response.json();
      console.log('\n❌ API CALL FAILED');
      console.log('Error details:', JSON.stringify(errorData, null, 2));
      
      if (response.status === 401) {
        console.log('\n⚠️  This usually means:');
        console.log('   - API key is invalid');
        console.log('   - API key has been revoked');
        console.log('   - API key format is incorrect');
      } else if (response.status === 429) {
        console.log('\n⚠️  This usually means:');
        console.log('   - Rate limit exceeded');
        console.log('   - Insufficient credits in your OpenAI account');
      }
      process.exit(1);
    }

    const data = await response.json();
    const message = data.choices[0].message.content;
    
    console.log('\n✅ API KEY IS WORKING!');
    console.log(`Response: ${message}`);
    console.log('\n✓ Your OpenAI API key is valid and working correctly.');
    console.log('✓ The study material generation should work now.');
    
  } catch (error) {
    console.log('\n❌ ERROR TESTING API');
    console.log('Error:', error.message);
    if (error.message.includes('fetch')) {
      console.log('\n⚠️  This might be a network issue. Check your internet connection.');
    }
    process.exit(1);
  }
}

testOpenAI();

